<?php
/**
 * Email Configuration for Zen Zone Spa
 * Uses environment variables for secure credential management
 * 
 * SETUP INSTRUCTIONS:
 * 1. Ensure backend/.env file exists with your actual credentials
 * 2. All sensitive data is loaded from .env automatically
 * 3. The fallback values below are placeholders only
 * 
 * SECURITY NOTE: NEVER commit credentials to this file!
 */

// Load environment configuration
require_once __DIR__ . '/env.php';

// Email Configuration using environment variables
// All sensitive values are loaded from .env file
// Fallback values are placeholders only - UPDATE YOUR .env FILE!
$emailConfig = [
    // SMTP Server Settings
    'smtp_server' => EnvLoader::get('SMTP_SERVER', 'smtp.example.com'),
    'smtp_port' => EnvLoader::getInt('SMTP_PORT', 465),
    'smtp_security' => EnvLoader::get('SMTP_SECURITY', 'ssl'),
    
    // Authentication (loaded from .env - DO NOT hardcode credentials here!)
    'smtp_username' => EnvLoader::get('SMTP_USERNAME', 'username@example.com'),
    'smtp_password' => EnvLoader::get('SMTP_PASSWORD', ''),
    
    // Email Addresses
    'from_email' => EnvLoader::get('FROM_EMAIL', 'noreply@example.com'),
    'from_name' => EnvLoader::get('FROM_NAME', 'Zen Zone Spa Contact Form'),
    'to_email' => EnvLoader::get('TO_EMAIL', 'admin@example.com'),
    
    // Email Templates
    'subject_template' => EnvLoader::get('SUBJECT_TEMPLATE', 'New Contact Inquiry - Zen Zone Spa'),
    'auto_reply_subject' => EnvLoader::get('AUTO_REPLY_SUBJECT', 'Thank you for contacting Zen Zone Spa'),
    
    // Settings
    'send_auto_reply' => EnvLoader::getBool('SEND_AUTO_REPLY', true),
    'log_emails' => EnvLoader::getBool('LOG_EMAILS', true),
    'fallback_to_basic_mail' => EnvLoader::getBool('FALLBACK_TO_BASIC_MAIL', true)
];

// Define the configuration constant
define('EMAIL_CONFIG', $emailConfig);

/**
 * Get email configuration
 */
function getEmailConfig() {
    return EMAIL_CONFIG;
}

/**
 * Validate email configuration
 */
function validateEmailConfig() {
    $config = EMAIL_CONFIG;
    $required = ['smtp_server', 'smtp_username', 'smtp_password', 'from_email', 'to_email'];
    
    foreach ($required as $field) {
        if (empty($config[$field])) {
            throw new Exception("Email configuration missing: $field");
        }
    }
    
    return true;
}

/**
 * Test email configuration
 */
function testEmailConfig() {
    try {
        validateEmailConfig();
        
        $config = EMAIL_CONFIG;
        
        // Test SMTP connection
        $connection = @fsockopen($config['smtp_server'], $config['smtp_port'], $errno, $errstr, 10);
        
        if (!$connection) {
            return [
                'success' => false,
                'message' => "Cannot connect to SMTP server: $errstr ($errno)"
            ];
        }
        
        fclose($connection);
        
        return [
            'success' => true,
            'message' => 'Email configuration is valid',
            'config' => [
                'server' => $config['smtp_server'],
                'port' => $config['smtp_port'],
                'security' => $config['smtp_security'],
                'from' => $config['from_email'],
                'to' => $config['to_email']
            ]
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => $e->getMessage()
        ];
    }
}

/**
 * Get environment status for debugging
 */
function getEnvironmentStatus() {
    $envFile = __DIR__ . '/../.env';
    
    return [
        'env_file_exists' => file_exists($envFile),
        'env_file_path' => $envFile,
        'env_file_readable' => file_exists($envFile) && is_readable($envFile),
        'loaded_vars' => [
            'SMTP_SERVER' => EnvLoader::has('SMTP_SERVER') ? 'Set' : 'Not set',
            'SMTP_USERNAME' => EnvLoader::has('SMTP_USERNAME') ? 'Set' : 'Not set',
            'SMTP_PASSWORD' => EnvLoader::has('SMTP_PASSWORD') ? 'Set (hidden)' : 'Not set',
            'FROM_EMAIL' => EnvLoader::has('FROM_EMAIL') ? 'Set' : 'Not set',
            'TO_EMAIL' => EnvLoader::has('TO_EMAIL') ? 'Set' : 'Not set'
        ],
        'using_defaults' => !EnvLoader::has('SMTP_SERVER')
    ];
}
