<?php
/**
 * Environment Configuration Loader
 * Simple .env file parser for secure configuration management
 */

class EnvLoader {
    private static $envVars = [];
    private static $loaded = false;
    
    /**
     * Load environment variables from .env file
     */
    public static function load($envPath = null) {
        if (self::$loaded) {
            return;
        }
        
        if ($envPath === null) {
            $envPath = __DIR__ . '/../.env';
        }
        
        // Try to load from .env file
        if (file_exists($envPath)) {
            $lines = file($envPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            
            foreach ($lines as $line) {
                // Skip comments
                if (strpos(trim($line), '#') === 0) {
                    continue;
                }
                
                // Parse key=value pairs
                if (strpos($line, '=') !== false) {
                    list($key, $value) = explode('=', $line, 2);
                    $key = trim($key);
                    $value = trim($value);
                    
                    // Remove quotes from value
                    $value = trim($value, '"\'');
                    
                    // Store in our array and set as environment variable
                    self::$envVars[$key] = $value;
                    $_ENV[$key] = $value;
                }
            }
        }
        
        self::$loaded = true;
    }
    
    /**
     * Get environment variable with fallback
     */
    public static function get($key, $default = null) {
        // Load env vars if not already loaded
        self::load();
        
        // Try environment variables first
        if (isset($_ENV[$key])) {
            return $_ENV[$key];
        }
        
        // Try our loaded vars
        if (isset(self::$envVars[$key])) {
            return self::$envVars[$key];
        }
        
        // Try system environment
        $value = getenv($key);
        if ($value !== false) {
            return $value;
        }
        
        // Return default
        return $default;
    }
    
    /**
     * Get boolean value
     */
    public static function getBool($key, $default = false) {
        $value = self::get($key, $default);
        
        if (is_bool($value)) {
            return $value;
        }
        
        if (is_string($value)) {
            $value = strtolower($value);
            return in_array($value, ['true', '1', 'yes', 'on']);
        }
        
        return (bool) $value;
    }
    
    /**
     * Get integer value
     */
    public static function getInt($key, $default = 0) {
        $value = self::get($key, $default);
        return (int) $value;
    }
    
    /**
     * Check if environment variable exists
     */
    public static function has($key) {
        self::load();
        return isset($_ENV[$key]) || isset(self::$envVars[$key]) || getenv($key) !== false;
    }
    
    /**
     * Get all loaded environment variables
     */
    public static function all() {
        self::load();
        return array_merge(self::$envVars, $_ENV);
    }
}

// Auto-load environment variables
EnvLoader::load();
